/////////////////////////////////////////////////////////////////////////////////

// Original obtained from ShaderToy.com
// Adapted, trivialy, for VGHD by TheEmu. TheEmu.

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// Use defines here rather than edit the body of the code.

#define iGlobalTime (u_Elapsed/10.0)
#define iResolution u_WindowSize

/////////////////////////////////////////////////////////////////////////////////

// If no funtion is selected then the screen will be divided into 20 cells, one
// for each function, which will be coloured and animated individualy.

/////////////////////////////////////////////////////////////////////////////////

// The ShaderToy shaders often use textures as inputs named iChannel0. With VGHD
// this may access a Sprite, ClipSprite or ClipNameSprite image depending on how
// the .scn file declares them.
//
// Note, the name used here does not seem to make any difference, so I have used
// iChannel0 which is what is used by ShaderToy but you can use any name as long
// as it matches the use in the main body of the shader. TheEmu.

uniform sampler2D iChannel0;

// With VGHD the range of the P argument's components of the texture functions is
// 0.0 to 1.0 whereas with ShaderToy it seems that the upper limits are given  by
// the number of pixels in each direction, typically 512 or 64.  We therefore use
// the following functions instead.

vec4 texture2D_Fract(sampler2D sampler,vec2 P) {return texture2D(sampler,fract(P));}
vec4 texture2D_Fract(sampler2D sampler,vec2 P, float Bias) {return texture2D(sampler,fract(P),Bias);}

// Rather than edit the body of the original shader we use use a define  here  to
// redirect texture calls to the above functions.

#define texture2D texture2D_Fract

/////////////////////////////////////////////////////////////////////////////////

// If the display is jerky then reduce the values the for iteration limits. This
// will reduce the calcultions needed for each pixel,  but at the expense of the
// quality of generated image.

#define MAX_ITERATIONS_DOLPHIN 6 // Originaly 11
#define MAX_ITERATIONS_INTERSECT_DOLPHIN 20 // Originaly 50
#define MAX_ITERATIONS_INTERSECT_WATER 6 // Originaly 12
#define MAX_ITERATIONS_SHADOW 6 // Originaly 25

/////////////////////////////////////////////////////////////////////////////////

// Created by David Bargo - davidbargo/2015
// License Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.

#define PI 3.14159265359
#define E  2.71828182845

vec2 sinz(vec2 c) 
{
    float a = pow(E, c.y);
    float b = pow(E,-c.y);
    return vec2(sin(c.x)*(a + b)*0.5, cos(c.x)*(a - b)*0.5);
}

vec2 cosz(vec2 c) 
{
    float a = pow(E, c.y);
    float b = pow(E,-c.y);
    return vec2(cos(c.x)*(a + b)*0.5, -sin(c.x)*(a - b)*0.5);
}

vec2 tanz(vec2 c) 
{
    float a = pow(E, c.y);
    float b = pow(E,-c.y);
    float cosx = cos(c.x);
    float sinhy = (a - b)*0.5;
    return vec2(sin(c.x)*cosx, sinhy*(a + b)*0.5)/(cosx*cosx + sinhy*sinhy);
}

vec2 logz(vec2 c) 
{
    return vec2(log(sqrt(dot(c, c))), atan(c.y, c.x));
}

vec2 sqrtz(vec2 c) 
{
    float n = c.x + sqrt(dot(c, c));
    return vec2(n, c.y)/sqrt(2.0*n);
}

vec2 exp2z(vec2 c) 
{
   return vec2(c.x*c.x - c.y*c.y, 2.*c.x*c.y);   
}

vec2 epowz(vec2 c) 
{
   return vec2(cos(c.y), sin(c.y))*pow(E, c.x);   
}

vec2 mulz(vec2 c1, vec2 c2)
{
    return c1*mat2(c2.x, -c2.y, c2.y, c2.x);
}

vec2 divz(vec2 n, vec2 d)
{
    return n*mat2(d.x, d.y, -d.y, d.x)/dot(d, d);
}

vec2 invz(vec2 c)
{
   return vec2(c.x, -c.y)/dot(c, c);
}

vec2 func(float cellID, vec2 c)
{
    vec2 fz = c;
    if      (cellID == 0.)  fz = c;
    else if (cellID == 1.)  fz = sinz(c);
    else if (cellID == 2.)  fz = sqrtz(divz(logz(vec2(-c.y - 6.0, c.x)), logz(vec2(-c.y + 2.0, c.x))));
    else if (cellID == 3.)  fz = epowz(c);
    else if (cellID == 4.)  fz = tanz(tanz(c));
    else if (cellID == 5.)  fz = tanz(sinz(c));
    else if (cellID == 6.)  fz = sqrtz(vec2(1.0 + c.x, c.y)) + sqrtz(vec2(1.0 - c.x, -c.y));
    else if (cellID == 7.)  fz = divz(tanz(exp2z(c)), c);
    else if (cellID == 8.)  fz = sinz(cosz(sinz(c)));
    else if (cellID == 9.)  fz = invz(vec2(1.0, 0.0) + epowz(vec2(c.y, c.x)));
    else if (cellID == 10.) fz = epowz(invz(sqrtz(-c)));
    else if (cellID == 11.) fz = exp2z(invz(c));
    else if (cellID == 12.) fz = epowz(sinz(epowz(cosz(c))));      
    else if (cellID == 13.) fz = divz(sinz(c), c);
    else if (cellID == 14.) fz = exp2z(c);
    else if (cellID == 15.) fz = divz(sinz(c), cosz(exp2z(c)));
    else if (cellID == 16.) fz = invz(c + vec2(1.0, 0.0)) + invz(c - vec2(1.0, 0.0));
    else if (cellID == 17.) fz = logz(c - invz(c));
    else if (cellID == 18.) fz = divz(sqrtz(vec2(c.x + 1.0, c.y)), sqrtz(vec2(c.x - 1.0, c.y)));
    else if (cellID == 19.) fz = invz(vec2(1.0, 0.0) + mulz(c, exp2z(exp2z(c))));

    return fz;   
}

vec2 animate(vec2 v) 
{
   float s = sin(iGlobalTime);
   float c = cos(iGlobalTime);
   return v*mat2(c, -s, s, c);
}

// iq's smooth hsv to rgb
vec3 hsv2rgb( in vec3 c )
{
    vec3 rgb = clamp( abs(mod(c.x*6.0+vec3(0.0,4.0,2.0),6.0)-3.0)-1.0, 0.0, 1.0 );
   rgb = rgb*rgb*(3.0-2.0*rgb);
   return c.z * mix( vec3(1.0), rgb, c.y);
}


vec2 gridSize = vec2(5.,4.);

vec3 getCell(vec2 s, vec2 h) 
{
    float cx = floor(h.x*gridSize.x/s.x);
    float cy = floor(h.y*gridSize.y/s.y);
    return vec3(cx, cy, (gridSize.y - 1.0 - cy)*gridSize.x + cx);
}

vec3 getSmallCells(vec2 s, vec2 h)
{
    vec3 c = getCell(s, h);
    vec2 size = s/gridSize;
    float ratio = size.x/size.y;
    vec2 uv = PI*((2.*h-size)/size.y - 2.*vec2(c.x*ratio,c.y));    
    return vec3(c.z, uv);
}

void mainImage( out vec4 fragColor, in vec2 fragCoord )
{
    vec2 res = iResolution.xy;
    vec2 coord = fragCoord.xy;

    vec3 cell = getSmallCells ( res, coord ); // Removed mouse dependancy. TheEmu

    vec2 z = animate(func(cell.x, cell.yz))*2.0;
    
#ifdef TEXTURIZE
    fragColor = texture2D(iChannel0, z*0.25);
#else 
    float hue = atan(z.y,z.x)/(2.0*PI);
    float l = length(z);
    float sat = abs(fract(l)-0.5)-0.25;
    sat = step(0.0, sat)*sat*4.0;
    sat = 1. - sat*sat;
    vec2 r = abs(fract(z) - 0.5) - 0.25;
    r = step(0.0, r)*r*4.0;
    r = 1. - r*r*r*r;
    float val = mix(1.0, r.x*r.y, sat*0.5);
    fragColor = vec4(hsv2rgb(vec3(hue,sat,val)),1.0);
#endif

}

void main ( void )
{
  mainImage( gl_FragColor, gl_FragCoord.xy );
}
